/** @file   LightSource.h
 * @brief   Declaration of LightSource class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/02/17 23:09:00 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_LIGHTSOURCEMODEL_H
#define H_WWW_LIGHTSOURCEMODEL_H

#include "Vec2D.h"
#include "datatypes.h"
#include "Color.h"
#include "www_exceptions.h"

namespace WeWantWar {

/** @class  LightSourceModel
 * @brief   A baseclass for all the lightsources.
 * @author  Tomi Lamminsaari
 *
 */
class LightSourceModel
{
public:
  ///
  /// Constants, datatypes and static methods
  /// =======================================
    
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Default constructor.
   * @param   aUid              Unique identifier. Every instance of
   *                            LightSourceModel must have unique id.
   * @exception Exception       Thrown if construction fails.
   */
  LightSourceModel( int aUid ) throw (Exception);

  /** Destructor
   */
  virtual ~LightSourceModel();

private:

  LightSourceModel( const LightSourceModel& aAnother );
  LightSourceModel& operator = ( const LightSourceModel& aAnother );

public:
  ///
  /// New public methods
  /// ==================
  
  /** Renders this lightsource model to certain position on the given
   * bitmap.
   * @param   aPosition         Coordinate where the lightsource should be
   *                            rendered.
   * @param   aCanvas           The bitmap where to draw.
   * @param   aIntensity        The intensity of the lightsource.
   */
  virtual void redraw( const eng2d::Vec2D& aPosition, BITMAP* aCanvas,
                       int aIntensity ) = 0;
  

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the unique identifier of this lightsource model.
   * @return  Id of this model.
   */
  int modelUid() const;
  
  
protected:
  ///
  /// Protected interface
  /// ===================


protected:
  ///
  /// Data members
  /// ============

  /** The unique identifier of this lightsource model. */
  int   iUid;
};



class PointLightModel : public LightSourceModel
{
public:  
  ///
  /// Constructors, destructor and operators
  /// ======================================
  
  /** Constructor.
   * @param   aUid              Unique identifier of this lightsource model
   * @param   aRadius           The radius of the light area in pixels.
   * @param   aColor            Color of the light.
   * @exception Exception       Thrown if construction fails.
   */
  PointLightModel( int aUid, int aRadius,  const eng2d::Color& aColor ) throw (Exception);
  
  /** Destructor.
   */
  virtual ~PointLightModel();

private:
  PointLightModel( const PointLightModel& aAnother );
  PointLightModel& operator = ( const PointLightModel& aAnother );
public:

  ///
  /// Inherited methods
  /// =================
  
  /** From LightSourceModel
   */
  virtual void redraw(const eng2d::Vec2D& aPosition, BITMAP* aCanvas,
                      int aIntensity );
                      
protected:
  ///
  /// Data members
  /// ============
  
  /** Pointer to bitmap that contains the graphic representation of the
   * light source.
   */
  BITMAP* iLightGfx;
};

};  // end of namespace

#endif
